## This script is used to generate population, wealth, and wealth transfer sizes at each year, each cohort and for each scenario 
## both with and without the existence of future wealth transfers.

## This script is not designed to be run standalone.
## Run "5 - Cohort simulation model - run scenarios.R" to draw on this script and generate the report results.
## Read Appendix C for a description of the processes here


# Additional preparation -------------------------------------------------------------

year_list <- copy(year_list_0) ## https://stackoverflow.com/questions/8030452/pass-by-reference-operator-in-the-data-table-package-modifies-another-data

## at year 0, add starting n incl parents and starting wealth
year_list[["0"]] <- merge(year_list[["0"]], starting_cohorts_n_wealth,
                          by = c("starting_cohort", "year"),
                          all.x=T)
## create variables for use in n-calculation process
year_list[["0"]] <- year_list[["0"]][, end_year_deaths:= 0] %>% 
  mutate(across(contains("parent_age0_n"), ~0, .names = "{.col}_deaths")) %>% 
  mutate(across(matches("parent_age0_n_.*_deaths"), ~0, .names="{.col}_add")) %>% 
  ## new variable and starting value for model lifetime inc
  mutate(model_lifetime_inc_2018=0) %>% 
  ## remove unneeded variables for year 0
  select(-c(age_grp2, trans_from, mortality_year, beqrec_0, ho_0, inc_trans_possible, inc_trans_year, inc_trans_num, inc_0, wealth_0,
            death_rate, income, netincome, marginal_tax, super_contrib, avprob, gift_giving_rate),
         -contains("param"))

## string identifying vars to draw previous year's data from during projection - n and wealth vars
prev_names <- "starting_cohort|^inc_pathway|^beqrec$|^ho$|n$|_n_|deaths|^housing_assets$|^super_assets$|^other_assets$|^housing_debt$|model_lifetime"

## names of vars relating to parents for use in model
SDcol_parent_deaths <- str_subset(names(year_list[[1]]), "parent_.*_deaths$")
SDcol_prev_parent_deaths <- paste0("prev_", SDcol_parent_deaths)
SDcol_parent_deaths_add <- paste0(SDcol_parent_deaths, "_add") ## additional parent deaths excluding deaths in own cohort
SDcol_prev_parent_deaths_add <- paste0("prev_", SDcol_parent_deaths_add)

SDcol_parent_n <- str_subset(names(year_list[[1]]), "parent_age0_n.*[\\)\\]]$")
SDcol_prev_parent_n <- paste0("prev_", SDcol_parent_n)
SDcol_parent_mort <- str_subset(names(bequest_receipt_prob), "parent_age0_mort.*[\\)\\]]$")

SDcol_prev_parent_deaths_add_forcurrent <- paste0(SDcol_prev_parent_deaths_add, "_forcurrent") ## additional parent deaths last year, correctly numbered for the current cohort
SDcol_parent_av_beqrec <- paste0("parent_age0_", str_extract(SDcol_parent_n, "\\[.*(\\)|\\])"), "_av_beqrec")
SDcol_parent_av_giftrec <- paste0("parent_age0_", str_extract(SDcol_parent_n, "\\[.*(\\)|\\])"), "_av_giftrec")

## initial bequest pool for year 1 (no bequests because no one died in year 0)
bequest_pool <- lapply(unique(year_list[[1]]$age_grp_0),
                       function(x) {
                         data.frame(
                           parent_age0=x,
                           deaths=0,
                           bequest_pool= 0
                         )
                       }) %>% 
  rbindlist

## initial parent to child ratios by parent age group
## ie for every age cohort "parent" in the data, how many children are there?
## used to modify gift analysis to account for the fact we have 0 births in the model.
parent_n <- merge(year_list_0[["0"]], starting_cohorts_n_wealth,
                  by = c("starting_cohort", "year"),
                  all.x=T) %>% 
  group_by(age_grp) %>% 
  summarise(parent_n=sum(n)) %>% 
  rename(parent_age_grp= age_grp)

child_w_parents_n <- merge(year_list_0[["0"]], starting_cohorts_n_wealth,
                           by = c("starting_cohort", "year"),
                           all.x=T) %>% 
  pivot_longer(cols=contains("parent_age0_n"), names_to="parent_age0_n", values_to="parents_alive") %>% 
  group_by(parent_age0_n) %>% 
  summarise(child_w_parents_alive_n = sum(parents_alive, na.rm=T)) %>% 
  mutate(parent_age_grp = str_extract(parent_age0_n, "\\[.*(\\)|\\])")) %>% 
  select(-parent_age0_n)

parent_child_ratio <- left_join(parent_n, child_w_parents_n) %>% 
  mutate(parent_child_ratio = parent_n/child_w_parents_alive_n)


## NOTE: parent variables have to be in the right order 



# Year loop ---------------------------------------------------------------

## generate population, wealth, and wealth transfer sizes at each year, each cohort 

for(i in c(2:length(year_list))) {
  gc()
  message("Year ", i-1)
  
  if(w_transfers==0 | w_gifts==0) { ## in no transfers scenarios, no gifts given
    year_list[[i]][, gift_giving_rate :=0 ]
  }
  
  if( (w_transfers==1 | w_bequests==1 ) & w_bequests_dont_affect_homeownership==0) { 
    ## increase probability of home ownership if received a bequest
    ## based roughly on Cigdem and Whelan which finds that bequests increase the probability of being in housing
    ## assume bequest receipt increases housing probability for 25-45 by 3.7 percentage points, and for 45-55 yos by 1 percentage point in next year onwards
    ## This is turned off in scenarios where we assume bequests do not affect homeownership 
    ## --- (2a where the sequencing of the model makes it too difficult to allow this to increase only for highest wealth people, because population size and wealth group is determined later)
    ## --- (2f where we cannot ensure that total benefit of inheritance into housing and other assets is not overstated if this is turned on)
    year_list[[i]][, home_trans_prob := case_when( beqrec==1 & age_grp >= "[25,30)" & age_grp <= "[40,45)" ~ home_trans_prob + 0.037,
                                                   beqrec==1 & age_grp >= "[45,50)" & age_grp <= "[55,60)" ~ home_trans_prob + 0.01,
                                                   TRUE ~ home_trans_prob) ]
  }
  
  ## get previous year values of n for each starting cohort, each possible previous inc pathway, each possible previous beqrec year and each possible previous ho group
  prev_n_wealth <- year_list[[i-1]][, .SD, .SDcols=patterns(prev_names)] %>% 
    ## rename variables
    setnames(paste0("prev_", names(.))) %>% 
    setnames(old="prev_starting_cohort", new="starting_cohort")
  
  ## add identifiers to current year data to allow merging to correct previous year values for n and deaths
  year_list[[i]][, prev_inc_pathway := ifelse(inc_trans_possible==T & inc_trans_year==T, 
                                              stri_sub(inc_pathway, 1, -2),
                                              inc_pathway)]
  year_list[[i]][, prev_beqrec := ifelse(beqrec==1, 1, 0) ] ## if= -1 or 0, then you hadn't received a bequest in previous year
  year_list[[i]][, prev_ho := ifelse(ho==1, 1, 0) ] ## as above for homeownership
  
  ## merge with previous year n and wealth data, and calc total parent deaths for those who had not received a bequest previous year
  year_list[[i]] <- merge(year_list[[i]], prev_n_wealth, 
                          by=c("starting_cohort", "prev_inc_pathway", "prev_beqrec", "prev_ho"),
                          all.x=T)
  ## remove impossible transitions (ie where they don't match to any of last year's data)
  year_list[[i]] <- year_list[[i]][!is.na(prev_n), ]
  year_list[[i]][, prev_parent_deaths := rowSums(.SD, na.rm=T), .SDcols = SDcol_prev_parent_deaths] # https://stackoverflow.com/questions/21857679/summing-across-rows-of-a-data-table-for-specific-columns
  
  ## UPDATE N VALUES
  ## calculate updated n and parent n for the year 
  ## - if was already homeowner and already received a bequest, it is previous n minus previous year deaths, times probability of being in this inc group
  ## - if was not homeowner and already received a bequest, some move into home ownership at given rate if it is a home transition year
  ## - if was already homeowner (or it's not a home transition year) and hadnt received a bequest, some will have received a bequest at start of year based on parent deaths
  ## - if was not homeowner and had not received a bequest, some move into home ownership if it is a home transition year and some receive bequest and some do both or neither
  year_list[[i]][, n := case_when(beqrec== 1 & ho== 1 ~ (prev_n - prev_end_year_deaths)*avprob ,
                                  
                                  beqrec== 1 & ho== -1 ~ (prev_n - prev_end_year_deaths)*avprob*home_trans_prob,
                                  beqrec== 1 & ho== 0 ~ (prev_n - prev_end_year_deaths)*avprob*(1-home_trans_prob),
                                  
                                  ## interpretation for 1st: of the ppl who didn't die last year and owned a home but not bequest, multiply by probability of being in this income group,
                                  ##     and multiply by the number of additional parent deaths (excluding those who "died" due to own death) divided by all people whose parents were still alive last year
                                  beqrec== -1 & ho== 1 ~ (prev_n - prev_end_year_deaths)*avprob*((prev_parent_deaths - prev_end_year_deaths)/(prev_n - prev_end_year_deaths)),
                                  beqrec== 0 & ho== 1 ~ (prev_n - prev_end_year_deaths)*avprob*(1-(prev_parent_deaths - prev_end_year_deaths)/(prev_n - prev_end_year_deaths)),
                                  
                                  beqrec== -1 & ho== -1 ~ (prev_n - prev_end_year_deaths)*avprob*home_trans_prob*((prev_parent_deaths - prev_end_year_deaths)/(prev_n - prev_end_year_deaths)), 
                                  beqrec== -1 & ho== 0 ~ (prev_n - prev_end_year_deaths)*avprob*(1-home_trans_prob)*((prev_parent_deaths - prev_end_year_deaths)/(prev_n - prev_end_year_deaths)),
                                  beqrec== 0 & ho==-1 ~ (prev_n - prev_end_year_deaths)*avprob*home_trans_prob*(1-(prev_parent_deaths - prev_end_year_deaths)/(prev_n - prev_end_year_deaths)),
                                  beqrec== 0 & ho== 0 ~ (prev_n - prev_end_year_deaths)*avprob*(1-home_trans_prob)*(1-(prev_parent_deaths - prev_end_year_deaths)/(prev_n - prev_end_year_deaths))
  )]
  
  ## remove 0 or NA n. 0s are due to 0 avprob (probability of transitioning down this inc path) or 0 home_trans_prob (prob of transitioning into home ownership). NAs are due to not having previous data to merge onto
  year_list[[i]] <- year_list[[i]][n>0.0000000001 & !is.na(n), ]
  
  ## number of parents still alive by NEW cohort (parent deaths calculation later includes deaths of people in the cohort --- need to multiply by probability of being in this cohort among people with alive parents
  year_list[[i]][beqrec==0, (SDcol_parent_n) := (.SD[, SDcol_prev_parent_n, with=F] - .SD[, SDcol_prev_parent_deaths, with=F]) *n/(prev_n - prev_parent_deaths) ]
  
  
  ## BEQUESTS RECEIVED
  ## number of parents of living cohorts who died last year. 
  ## This equation: sum of SDcol_prev_parent_deaths_add (additional parent deaths) applies to everyone with the same previous transition path - need to apportion it to people in current transition path
  ## Note (prev_parent_deaths - prev_end_year_deaths) = number of people whose parents died because prev_parent_deaths includes own cohort deaths
  year_list[[i]][beqrec==-1, (SDcol_prev_parent_deaths_add_forcurrent) := (.SD[, SDcol_prev_parent_deaths_add, with=F])*n/(prev_parent_deaths - prev_end_year_deaths) ]
  year_list[[i]][beqrec==-1, prev_parent_deaths_add_forcurrent := n] ## this will be used for summarising later - number of ppl who received bequest this year
  
  ## CHECK total ADDITIONAL PARENT DEATH NUMBERS IS CONSISTENT with prev year - yes
  # sum(year_list[[i]] %>% select(contains("forcurrent")), na.rm=T)
  # sum(year_list[[i-1]] %>% select(matches("parent.*deaths_add")), na.rm=T)
  
  ## calculate distributions from the bequest pools based on ages of parents who died
  av_bequest_received <- year_list[[i]] %>% 
    ## total number of living cohorts who had parents who died by parent age grp
    summarise(across(all_of(SDcol_prev_parent_deaths_add_forcurrent), ~sum(.x, na.rm=T) )) %>% 
    pivot_longer(cols = contains("forcurrent"), names_to="parent_age0", values_to="people_parents_died") %>% 
    mutate(parent_age0 = str_extract(parent_age0, "\\[.*(\\)|])") ) %>% 
    ## merge with bequest pool data
    left_join(bequest_pool) %>% ## comparing number of people whose parents died vs parent deaths - reasonable proportions except for approx parents aged 90+ and under 25
    ## rough adjustment - total bequest pool size to be distributed is no larger than would be implied if twice the people whose parents died (2 parents). Mainly relevant for ppl who died young - less likely to have children to bequest to
    mutate(bequest_pool = ifelse(people_parents_died>0 & deaths/people_parents_died>2, bequest_pool/deaths*people_parents_died*2, bequest_pool)) %>% 
    ## calculate average bequest received by people whose parents died
    mutate(av_beqrec = ifelse(people_parents_died>0, bequest_pool/people_parents_died, 0) ) %>% 
    ## pivot wide to merge back with year list
    select(parent_age0, av_beqrec) %>% 
    mutate(year= i-1,
           parent_age0 = paste0("parent_age0_", parent_age0, "_av_beqrec")) %>% 
    pivot_wider(names_from = parent_age0, values_from = av_beqrec)
  
  ## pool those distributions from parent pools into a different set of pools based on own age group
  bequest_pool_own_age <- year_list[[i]] %>% 
    ## total number of cohorts who had parents who died by parent age grp
    group_by(age_grp) %>% 
    summarise(across(all_of(SDcol_prev_parent_deaths_add_forcurrent), ~sum(.x, na.rm=T) )) %>% 
    mutate(year= i-1) %>% 
    ## merge in av beqrec by parent age 
    left_join(av_bequest_received) %>% 
    data.table
  ## and calc total own-age bequest pools and total parents died
  bequest_pool_own_age[, bequest_pool_own_age := rowSums(.SD[, SDcol_prev_parent_deaths_add_forcurrent, with=F] * .SD[, SDcol_parent_av_beqrec, with=F] , na.rm=T ) ]
  bequest_pool_own_age[, c("year", SDcol_prev_parent_deaths_add_forcurrent, SDcol_parent_av_beqrec) := NULL]
  
  ## merge bequest pool own age into year_list data
  year_list[[i]] <- merge(year_list[[i]], bequest_pool_own_age, by=c("age_grp"), all.x=T)
  ## create age-specific wealth group variable for bequest recipients and the proportion of bequest pool each group gets
  year_list[[i]] <- split(year_list[[i]], year_list[[i]]$age_grp)
  year_list[[i]] <- lapply(year_list[[i]], function(x) {
    x[, prev_total_wealth := prev_housing_assets + prev_super_assets + prev_other_assets - prev_housing_debt]
    
    if(w_2c_anticipation_effect==0) { ## Note in year 1, there are no beqrec==-1
      x[beqrec==-1, prev_age_wlth_grp3 := ifelse(age_grp>="[30,35)" & age_grp<="[85,90)",
                                                 cut(round(prev_total_wealth, 2), 
                                                     ## cut points determined by weighted quantile (only works if cut points are unique)
                                                     breaks = wtd.quantile(x$prev_total_wealth, ## quantile across everyone, not just beqrec
                                                                           weights = x$n, 
                                                                           probs = seq(0, 1, 1/3)) %>% round(2), 
                                                     include.lowest=T,
                                                     labels = c(1:3),
                                                     ordered_result=T),
                                                 0)]
      ## fixes where it sometimes cuts off at max or min ---- (note only applies where NA - ages 30-90 and beqrec==-1)
      x[beqrec==-1 , 
        prev_age_wlth_grp3 := case_when(is.na(prev_age_wlth_grp3) & round(prev_total_wealth, 2) == round(min(prev_total_wealth), 2) ~ 1,
                                        is.na(prev_age_wlth_grp3) & round(prev_total_wealth, 2) == round(max(prev_total_wealth), 2) ~ 3,
                                        TRUE ~ as.numeric(prev_age_wlth_grp3)
      ) ]
    }
    
    ## in this scenario we need to calc quantile by beqrec 0 too
    ## Note in year 1 (i=2), there are not enough unique cut points at some ages. 
    ## So don't apply this in year 1- have a constant anticipation effect instead - make same as mid wealth
    if(w_2c_anticipation_effect==1 & w_transfers==1 & i!=2) {
      x[beqrec %in% c(0,-1), prev_age_wlth_grp3 := ifelse(age_grp>="[30,35)" & age_grp<="[85,90)",
                                                 cut(round(prev_total_wealth, 2), 
                                                     ## cut points determined by weighted quantile (only works if cut points are unique)
                                                     breaks = wtd.quantile(x$prev_total_wealth, ## quantile across everyone, not just beqrec
                                                                           weights = x$n, 
                                                                           probs = seq(0, 1, 1/3)) %>% round(2), 
                                                     include.lowest=T,
                                                     labels = c(1:3),
                                                     ordered_result=T),
                                                 0)]
      ## fixes where it sometimes cuts off at max or min
      x[beqrec %in% c(0,-1), prev_age_wlth_grp3 := case_when(is.na(prev_age_wlth_grp3) & round(prev_total_wealth, 2) == round(min(prev_total_wealth), 2) ~ 1,
                                                    is.na(prev_age_wlth_grp3) & round(prev_total_wealth, 2) == round(max(prev_total_wealth), 2) ~ 3,
                                                    TRUE ~ as.numeric(prev_age_wlth_grp3)
      ) ]
    }
    if(w_2c_anticipation_effect==1 & w_transfers==1 & i==2) {
      x[beqrec %in% c(0,-1), prev_age_wlth_grp3 := 2]
    }
    
    ## create variable of total parent deaths within age grp, and within age and wlth grp
    x[beqrec==-1, prev_parent_deaths_own_age := sum(prev_parent_deaths_add_forcurrent, na.rm=T)]
    x[beqrec==-1, prev_parent_deaths_own_age_wlth := sum(prev_parent_deaths_add_forcurrent, na.rm=T), by=prev_age_wlth_grp3]
    
    ## Note number of people in quantiles won't be exactly equal due to cohort nature of the model and diffs in parent deaths. Adjust dists from bequest pool accordingly
    ## merge in distributions which is based on 1/3 in each age_wlth_grp3
    age_specific_beq_dist_adj <- distinct(x, beqrec, prev_age_wlth_grp3, prev_parent_deaths_own_age, prev_parent_deaths_own_age_wlth) %>% 
      filter(!is.na(prev_age_wlth_grp3) & beqrec==-1) %>% 
      left_join(age_wlth_beq_dist, by=c("prev_age_wlth_grp3")) %>% 
      ## adjust distributions based on actual proportions of people whose parents died within each wlth grp
      mutate(age_wlth_bequest_dist = age_wlth_bequest_dist/(1/3) * prev_parent_deaths_own_age_wlth/prev_parent_deaths_own_age,
             ## rescale to 1
             age_wlth_bequest_dist = age_wlth_bequest_dist/sum(age_wlth_bequest_dist)) %>% 
      select(beqrec, prev_age_wlth_grp3, age_wlth_bequest_dist) 
    
    x <- merge(x, age_specific_beq_dist_adj, by=c("prev_age_wlth_grp3", "beqrec"), all.x=T)
    
    x
  }) %>% 
    rbindlist(use.names=T)
  
  ## Calculate average bequest received = bequest pool for your age * share that your wlth grp gets / total parent deaths within age-wlth grp
  year_list[[i]][, av_beqrec := ifelse(beqrec==-1,
                                       bequest_pool_own_age * age_wlth_bequest_dist / prev_parent_deaths_own_age_wlth,
                                       0)]
  
  ## scenario for 2b: Wealthy save more of inheritance -- replace bequest saving rates to use
  if(w_2b_wealthy_save_more==1 ) {
    year_list[[i]] <- merge(year_list[[i]], bequest_saving_age_wlth_grp3, by=c("beqrec", "prev_age_wlth_grp3"), all.x=T)
    year_list[[i]][beqrec==-1, bequest_other_asset_saving_param := bequest_other_asset_saving_grp3_param ]
    year_list[[i]][, bequest_other_asset_saving_grp3_param :=NULL ]
  }
  
  ## scenario for low wealth saving less before bequest -- replace other asset saving rates to use (age 30-90)
  if(w_2c_anticipation_effect==1 & w_transfers==1) {
    year_list[[i]][beqrec==0, other_asset_saving_param := case_when(prev_age_wlth_grp3==1 ~ 0.6*other_asset_saving_param,
                                                                    prev_age_wlth_grp3==2 ~ 0.8*other_asset_saving_param,
                                                                    prev_age_wlth_grp3==3 ~ 0.9*other_asset_saving_param,
                                                                    TRUE ~ other_asset_saving_param)]
  }
  
  ## GIFTS GIVEN AND RECEIVED
  if(w_2f_housing_for_gifts==0) {
    year_list[[i]][, av_gift_given := prev_other_assets*gift_giving_rate]
  }
  if(w_2f_housing_for_gifts==1) { 
    year_list[[i]][, av_gift_given := prev_other_assets*gift_giving_rate + prev_housing_assets*housing_for_gifts_rate]
  }
  
  gift_pool <-  lapply(unique(year_list[[i]]$age_grp_0),
                       function(x) {
                         data.table(
                           parent_age0= x,
                           parent_age_grp= year_list[[i]][age_grp_0==x, age_grp][1],
                           parent_age0_n = with(year_list[[i]][age_grp_0==x, ], sum(n)), ## note not all have children / gifting. this is based on average gifting rate
                           gift_pool= with(year_list[[i]][age_grp_0 ==x, ], 
                                           sum( n*(av_gift_given) ))
                         )
                       }) %>% 
    rbindlist 
  
  av_gift_received <- year_list[[i]] %>% 
    ## total number of living cohorts who have parents who are still alive
    summarise(across(all_of(SDcol_parent_n), ~sum(.x, na.rm=T) )) %>% 
    pivot_longer(cols = contains("age0_n"), names_to="parent_age0", values_to="people_parents_alive") %>% ## interpretation: number of people who have parents aged 60-65 etc
    mutate(parent_age0 = str_extract(parent_age0, "\\[.*(\\)|])") ) %>% 
    ## merge with gift pool data
    left_join(gift_pool) %>% 
    ## merge with 2018 parent to child ratios
    left_join(parent_child_ratio) %>% 
    ## ratio of parents to kids gets infeasibly high for older age grps
    mutate(n_ratio = parent_age0_n/people_parents_alive) %>% 
    ## if model ratio is greater than 2018 ratio (because of no new births) and parent_age0 is <60, adjust the gift pool for dist to model people by the 2018 parent_child_ratio
    ## (excess is effectively assumed to be distributed to new births not in the model)
    mutate(gift_pool = gift_pool * parent_child_ratio/n_ratio) %>% 
    ## calculate average gift received by people whose parents are alive
    mutate(av_giftrec = ifelse(people_parents_alive>0, gift_pool/people_parents_alive, 0) ) %>% 
    ## pivot wide to merge back with year list
    select(parent_age0, av_giftrec) %>% 
    mutate(year= i-1,
           parent_age0 = paste0("parent_age0_", parent_age0, "_av_giftrec")) %>% 
    pivot_wider(names_from = parent_age0, values_from = av_giftrec)
  
  ## merge av giftrec values into year_list data
  year_list[[i]] <- merge(year_list[[i]], av_gift_received, by="year", all.x=T)
  ## calculate weighted average gift received (weighted by parents alive) by ea cohort - will apply only to beqrec=0 ie parents still alive
  year_list[[i]][, av_giftrec := ifelse(beqrec==0, 
                                        rowSums( .SD[, SDcol_parent_n, with=F ]*.SD[, SDcol_parent_av_giftrec, with=F] , na.rm=T) / 
                                          rowSums(.SD[, SDcol_parent_n, with=F ], na.rm=T),
                                        0)]
  
  ## WEALTH ACCUMULATION CALCULATIONS FOR EACH COHORT
  
  ## HOUSING wealth accumulation 
  if(w_2f_housing_for_gifts==0) { ## people who already owned housing have it increase by assumed rate of return and saving/drawdown param (as a fn of existing housing)
    year_list[[i]][ho ==1 , housing_assets := prev_housing_assets * (1+housing_asset_return_param + housing_asset_saving_param) ] 
  }
  if(w_2f_housing_for_gifts==1) { 
    year_list[[i]][ho ==1 , housing_assets := prev_housing_assets * (1-housing_for_gifts_rate) * (1+housing_asset_return_param + housing_asset_saving_param) ] 
  }
  year_list[[i]][ho ==-1 , housing_assets := new_housing_val_param] ## people who moved into housing this year have housing equal to housing starting value for their cohort
  year_list[[i]][ho ==0 , housing_assets := 0] ## people who stay non-homeowners have 0 housing
  

  year_list[[i]][ho ==1 , housing_debt := ifelse(housing_asset_saving_param>0,
                                                 pmax(0, prev_housing_debt*(1-housing_debt_reduc_param) + prev_housing_assets*housing_asset_saving_param*housing_debt_to_asset_param),
                                                 pmax(0, prev_housing_debt*(1-housing_debt_reduc_param)) )
                                                 ] ## people who were already in housing have their debt reduce by assumed rates, and increase by extra housing saving. housing drawdowns assumed not to reduce debt
  year_list[[i]][ho ==-1 , housing_debt := new_housing_val_param*housing_debt_to_asset_param] ## people who move into housing this year have debt as a proportion of new housing value
  year_list[[i]][ho ==0 , housing_debt := 0] ## people who stay non-homeowners have 0 housing debt
  
  
  ## SUPER wealth accumulation
  year_list[[i]][, super_assets := prev_super_assets*(1+super_asset_return_param)*(1-super_asset_drawdown_param) + super_contrib] ## super returns, minus drawdowns, plus new super
  ## lump sum drawdowns: if discounted value of super assets at the start of the period + discounted netincome is < 20k and cohort is in retirement phase, assume drawn down all super as lump sum except for new contributions
  year_list[[i]][age_grp>="[65,70)" & (prev_super_assets + netincome)/awe[year==i+2018-1, awe_growth_factor_2018] <20000, 
                 super_assets := super_contrib]
  
  ## OTHER wealth accumulation - assuming returns reinvested back into other assets
  year_list[[i]][, other_assets := prev_other_assets*(1-gift_giving_rate)*(1+other_asset_return_param*(1-marginal_tax))*(1-other_asset_drawdown_param) + ## other assets minus gifts, with returns taxed, minus drawdowns, 
                   ifelse(netincome>0, netincome*other_asset_saving_param, 0) + 
                   av_beqrec*bequest_other_asset_saving_param +
                   av_giftrec*gift_other_asset_saving_param ] ## plus new saving in other assets (as a fn of netincome and bequest and gift received)
  
  
  ## replace 0 av_beqrec and av_giftrec with NAs to exclude them from summaries below
  year_list[[i]][ beqrec!=-1, av_beqrec := NA] ## unless you received a bequest this year, make it NA
  year_list[[i]][ beqrec!=0, av_giftrec := NA] ## unless your parents are still alive, make it NA
  
  ## prev wealth of new bequest recips - to enable calcs of beq as a fn of prev wealth
  ## determine prev wealth for people who just received a bequest. calc variable for new beqrec only and then merge into ultimate summary
  year_list[[i]][ beqrec==-1, prev_total_wealth := prev_housing_assets + prev_super_assets + prev_other_assets - prev_housing_debt]
  
  ## calc for new beqrecs their prev wealth and av_beqrec, for merging into ultimate summary
  new_beqrec_vars <- year_list[[i]][beqrec==-1, c("starting_cohort", "age_grp_0", "age_grp", "total_inc_qtile", "inc_pathway", "beqrec", "ho", "n", "av_beqrec", "prev_total_wealth")]
  new_beqrec_vars <- new_beqrec_vars %>% 
    mutate(beqrec = ifelse(beqrec==-1, 1, beqrec),
           ho = ifelse(ho==-1, 1, ho)) %>%
    group_by(starting_cohort, age_grp_0, age_grp, total_inc_qtile, inc_pathway, beqrec, ho) %>% 
    summarise(av_beqrec = wtd.mean(av_beqrec, weights=n, na.rm=T), 
              prev_wealth_new_beqrec = wtd.mean(prev_total_wealth, weights = n))
  
  ## calc 2018 net income for adding onto lifetime income
  year_list[[i]][, netincome_2018 := netincome/awe[year==i+2018-1, awe_growth_factor_2018]]

  ## summarise n and wealth values by 0/1 beqrec and 0/1 ho (remove distinction between -1 and 1)
  year_list[[i]][, beqrec := ifelse(beqrec==-1, 1, beqrec)]
  year_list[[i]][, ho := ifelse(ho==-1, 1, ho)]
  
  year_list[[i]] <- year_list[[i]] %>% 
    group_by(starting_cohort, age_grp_0, age_grp, total_inc_qtile, inc_pathway, beqrec, ho) %>% 
    summarise(across(matches("^housing_assets$|^super_assets$|^other_assets$|^housing_debt$"), ~wtd.mean(.x, weights = n, na.rm=T) ), ## this needs to come before generation of new n
              
              total_wealth_at_death = housing_assets + super_assets + other_assets - housing_debt, ## calc this before living gens receive first estates
              ## wealth at death by type
              housing_assets_at_death = housing_assets,
              super_assets_at_death = super_assets,
              other_assets_at_death = other_assets,
              housing_debt_at_death = housing_debt,
              
              prev_parent_deaths_add = sum(prev_parent_deaths_add_forcurrent, na.rm=T), ## number of ppl who received a bequest this year (ie parent died last year)
              
              av_giftrec = wtd.mean(av_giftrec, weights=n, na.rm=T), ## should appear for beqrec=0 only. weighted av across homeowner 1 and -1 status
              av_giftgiven = wtd.mean(av_gift_given, weights=n, na.rm=T), 
              ## av_beqrec gets merged in later
              
              model_lifetime_inc_2018 = mean(prev_model_lifetime_inc_2018 + netincome_2018), ## total income received through the model (mean will be same across cohort because same age and inc path)
              
              across(matches("(^n$)|(^parent_age0_n)"), ~sum(.x, na.rm=T)), ## this needs to come towards the end to avoid errors in weights used above
              
              year= i-1, ## year is the same across everyone
              death_rate = mean(death_rate) ## death rate will be the same across age
    ) %>% 
    left_join(new_beqrec_vars) %>% ## add in av_beqrec and prev_wealth for new beqrecs
    data.table
  
  ## now calculate end of period deaths in each cohort and parent deaths by age
  year_list[[i]][, end_year_deaths := n*death_rate]
  
  ## parent deaths includes - "deaths" due to death of cohort, and additional deaths of parents for living cohort - assuming even distributions of parents across dead and living
  year_list[[i]] <- merge(year_list[[i]], bequest_receipt_prob, by=c("starting_cohort", "year"), all.x=T)
  ## equation: parent_n*death_rate + parent_n*(1-death_rate)*parent_death_rate = parent deaths due to cohort deaths + additional parent deaths
  year_list[[i]][, (SDcol_parent_deaths) := .SD[, SDcol_parent_n, with=F]* (death_rate + (1-death_rate)*(.SD[, SDcol_parent_mort, with=F]))  ]
  year_list[[i]][, (SDcol_parent_deaths_add) := .SD[, SDcol_parent_n, with=F]* (1-death_rate)*(.SD[, SDcol_parent_mort, with=F])  ] ## additional parent deaths only (for bequest calcs)
  
  ## remove columns from year_list
  remove_cols <- str_subset(names(year_list[[i]]), "mort")
  year_list[[i]][, (remove_cols) :=NULL ]
  
  ## create bequest pools by parent age for next period - less 2% passed to charity rather than other generations
  if(w_transfers==1 | w_bequests==1) {
    bequest_pool <- lapply(unique(year_list[[i]]$age_grp_0),
                           function(x) {
                             data.table(
                               parent_age0= x,
                               parent_age_grp= year_list[[i]][age_grp_0==x, age_grp][1],
                               deaths= with(year_list[[i]][age_grp_0==x, ], sum(end_year_deaths)),
                               bequest_pool= with(year_list[[i]][age_grp_0==x, ], 
                                                  sum( end_year_deaths*total_wealth_at_death ))
                             )
                           }) %>% 
      rbindlist %>% 
      ## merge in partnered status
      left_join(partnered_status) %>% 
      ## split bequest pool into portion that stays in same generation (partnered %) and portion that gets passed onto next generation (non-partnered %)
      mutate(bequest_pool_same_gen = bequest_pool*partnered_pc,
             bequest_pool = bequest_pool*(1-partnered_pc)*0.98 ) ## bequest_pool variable is for next gen. 
  }
  
  if(w_transfers==0 | w_bequests==0) {
    ## counterfactual: partners continue to get the wealth of their spouse, but none is passed intergenerationally
    
    bequest_pool <- lapply(unique(year_list[[i]]$age_grp_0),
                           function(x) {
                             data.table(
                               parent_age0= x,
                               deaths= with(year_list[[i]][age_grp_0==x, ], sum(end_year_deaths)),
                               bequest_pool= 0
                             )
                           }) %>% 
      rbindlist 
  }
  
  ## add bequested wealth from partners to the same cohort's wealth - assume it stays exactly in the same asset type, except super becomes other assets
  ## for each asset type -- add total amount left by the dead, split evenly between the living
  year_list[[i]] <- merge( year_list[[i]], partnered_status %>% rename(age_grp = parent_age_grp), by= "age_grp", all.x=T)
  year_list[[i]][, housing_assets := ifelse(n!=end_year_deaths, 
                                            housing_assets + partnered_pc*housing_assets*end_year_deaths/(n-end_year_deaths) , 
                                            housing_assets ) ]
  year_list[[i]][, other_assets := ifelse(n!=end_year_deaths, 
                                          other_assets + partnered_pc*(other_assets + super_assets) *end_year_deaths/(n-end_year_deaths),
                                          other_assets ) ]
  year_list[[i]][, housing_debt := ifelse(n!=end_year_deaths, 
                                          housing_debt + partnered_pc*housing_debt*end_year_deaths/(n-end_year_deaths),
                                          housing_debt) ]
  
  ## remove columns from year_list - this will run without error even though w_transfers=0 has no partnered_pc
  remove_cols <- c("death_rate")
  year_list[[i]][, (remove_cols) :=NULL ]
  
}



### TOTAL CHECKS ------------------------------------


# ## FULL check total n after deaths for all years
# sum_n <- sapply(year_list, function(x) sum(x$n))
# sum_deaths <- sapply(year_list, function(x) sum(x$end_year_deaths))
# sum_n[1:31] - sum_deaths[1:31] - sum_n[2:32] ## should equal 0. if -ve, then next period's n is greater than n-deaths
# 
# ## FULL check total n for parents for all years
# sum_n_beqrec0 <- sapply(year_list, function(x) sum(x[beqrec==0, n]))
# sum_n_parents <- sapply(year_list, function(x) sum(x[beqrec==0, SDcol_parent_n, with=F], na.rm=T))
# sum_n_beqrec0 - sum_n_parents ## should equal 0
# 
# ## FULL check parent n after deaths
# sum_parent_deaths <- sapply(year_list, function(x) sum(x[, SDcol_parent_deaths, with=F], na.rm=T))
# sum_n_parents[1:31] - sum_parent_deaths[1:31] - sum_n_parents[2:32] ## should equal 0
# 
# ## FULL check cohort deaths and parent additional deaths sum to total parent deaths 
# sum_parent_deaths_add <- sapply(year_list, function(x) sum(x[, SDcol_parent_deaths_add, with=F], na.rm=T))
# sum_deaths_w_parents <- sapply(year_list, function(x) sum(x[beqrec==0, "end_year_deaths", with=F])) ## deaths of ppl who have parents
# sum_deaths_w_parents + sum_parent_deaths_add - sum_parent_deaths ## should equal 0
# 
# 
# ## FULL check - any missing wealth data? 
# missing_wealth <- sapply(year_list, function(x) {
#   x %>%   
#     mutate(total_wealth = housing_assets + super_assets + other_assets - housing_debt) %>% 
#     filter(is.na(total_wealth)) %>% 
#     nrow
# })

